/*
 * Copyright (C) 2017 A Little Slice of MangOH and Renfell Engineering Pty Ltd - All Rights Reserved
 *
 * Use of this work is subject to license and solely for use with hardware
 * supplied by Renfell Engineering Pty Ltd.
 *
 * Proprietary and confidential.
 * This file or any part thereof may not be used, distributed, copied, or
 * modified for use for any purpose other than that for which it was
 * originally provided without first receiving written permission of the
 * copyright holder.
 *
 * This source code is provided 'as is' without warranty of any kind, either
 * expressed or implied by consumer legislation or otherwise, including but
 * not limited to the implied warranties of merchantability and/or fitness
 * for a particular purpose.
 *
 * The end user acknowledges that this software is a "work in progress", and
 * as such should not be relied upon in a commercial environment or other
 * situations where economic loss or harm to persons may arise due to a
 * failure to perform to the end users specific requirements.
 *
 * By use of this software you agree to the use of the above terms.
 *
 */


/**
 * Simple 'Master' for the Arduino Slave for the mangOH green DV3/DV4 board
 *
 * This application:
 *  sets up /dev/ttyUSB0 57600, N,8,1
 *  Sets up a timer to send a character to the serial port once per second
 *  Establishes a FD monitor to react when a character is available on the serial port
 *    when a character is available on the serial port, read it and display on 
 *     debug console
 *
 * Requirements:
 *  Sierra Wireless MangOH DV3 or DV3 board with legato V16.10.1 installed
 *  The onboard Arduino Leonardo is programmed with matching sample code
 *   
 */

#include "legato.h"
#include "interfaces.h"	/* include auto-generated api interfaces from Component.cdef */

#include <termios.h>

// this is for a DV3/4 board - arduino Leonardo on /dev/ttyUSB0
#define ARDUINO_SERIAL_PORT "/dev/ttyUSB0"

/* Global Variables */

const int32_t PollTimerMsPeriod = 1000;	// how often to poll the arduino with a new character

int32_t ArduinoFd = -1;

le_fdMonitor_Ref_t ArduinoFdMonitor = NULL;

le_timer_Ref_t PollTimerHandle = NULL;	// timer to poll the arduino

//-------------------------------------------------------------------------------------------------
/**
 * Init the Serial port parameters the *Linux* way
 */
//-------------------------------------------------------------------------------------------------

int32_t initSerialPort( int pFd )
{
	struct termios options;
	
	LE_INFO( "Configuring Fd [%d]", pFd );
	
	/* Get current options for the fd */
	tcgetattr( pFd, &options );

	/* make the device 'raw' */
	cfmakeraw( &options );
	
	/* set the baud rate to 57600 */
	cfsetspeed( &options, B57600);

	/* and set the new options to the fd */
	tcsetattr(pFd, TCSANOW, &options);
			  
	return pFd;
}

//-------------------------------------------------------------------------------------------------
/**
 * Timer handler to poll arduino with a new characterfor status updates
 *	This is a one-shot timer, so be sure to restart it if requred
 */
//-------------------------------------------------------------------------------------------------

void pollTimerHandler( le_timer_Ref_t pTimerRef )
{
	static char outChar = 'A';
	
	le_timer_Stop( PollTimerHandle );

	if ( ArduinoFd > -1 )	// idiot trap - only WRITE() if FD is open
	{
		int res = 0;
		res = write(ArduinoFd, &outChar, 1);		// write a simgle character to Arduino
		if ( 1 == res )								// sucessful write
		{
			LE_INFO("Timer: TX Char [%c] (0x%02x)", outChar, outChar);
		}
		else
		{
			LE_WARN("Timer: Error: res != 1 (%d) errno[%d][%s]", res, errno, strerror(errno));
		}
		outChar++;							// increment next char
		if ( outChar > 'z' ) 				// check for wraparound
		{
			outChar = 'A';
		}
		else if ((outChar > 'Z') && (outChar < 'a'))
		{
			outChar = 'a';
		}
	}

	// restart the poll timer
	LE_WARN_IF( LE_OK != (le_timer_SetRepeat( PollTimerHandle, 1)), "Arduino PollTimer SetRepeat failed");	// repeat once
	LE_WARN_IF( LE_OK != (le_timer_SetMsInterval( PollTimerHandle, PollTimerMsPeriod )), "Arduino PollTimer SetMsInterval failed");//);
	LE_WARN_IF( LE_OK != (le_timer_Start( PollTimerHandle )), "Arduino PollTimer Start Failed");
}

//-------------------------------------------------------------------------------------------------
/**
 * Handle READ() events on Arduino Serial Port
 * Read a char and output it on LE_INFO()
 */
//-------------------------------------------------------------------------------------------------

static void arduinoFdMonitorHandler(int pFd, short pEvents)
{
	if (pEvents & POLLIN )	// data is available for read
	{
		ssize_t len; 

		// need to read characters until no more data
		for (;;)	// read all data
		{
			char inChar;

			// read only one character at a time
			len = read(pFd, &inChar, 1);	// only read one character
			
			// test for error - if any error abort application
			if ((len == -1) && (errno != EAGAIN))
			{
				LE_FATAL("read read() error: [%d:%s]", errno, strerror(errno));
			}

			// if nothing to read, len==-1 && errno==EAGAIN. bail out
			if ( len <= 0 )
			{
				break;	// break out of read() loop
			}

			// now print out character
			LE_INFO("RX char [%c] (0x%02x)", inChar, inChar );
		}
	}
}

//-------------------------------------------------------------------------------------------------
/**
 * Signal Handler: called when SIGTERM signal received
 *	This lets us clean up nicely
 */
//-------------------------------------------------------------------------------------------------

static void sigHandlerSigTerm( int pSigNum )
{
	LE_INFO("SIGTERM caught, closing ArduinoMaster");
	
	if ( ArduinoFdMonitor != NULL )
	{
		le_fdMonitor_Delete( ArduinoFdMonitor );
		ArduinoFdMonitor = NULL;
	}

	if ( ArduinoFd > -1 )
	{
		close(ArduinoFd);
		ArduinoFd = -1;
	}
}



COMPONENT_INIT
{
	bool isInitialised = false;

    LE_INFO( "Arduino Master Init" );

    // setup to catch application termination and shutdown cleanly
    le_sig_Block( SIGTERM );
    le_sig_SetEventHandler( SIGTERM, sigHandlerSigTerm );

	do
	{		
		// init subsystems

		// first, open the serial port

		if ( (ArduinoFd = open(ARDUINO_SERIAL_PORT, (O_RDWR | O_NOCTTY /*| O_SYNC  */ | O_NONBLOCK /* | O_NDELAY */ ))) < 0 )
		{
			LE_WARN("Error opening %s err[%d]:%s", ARDUINO_SERIAL_PORT, errno, strerror(errno));
			break;
		}

		LE_INFO( "Open %s OK. fd[%d]",ARDUINO_SERIAL_PORT, ArduinoFd );

		// configure UART port using termios
		// see first answer at stackoverflow.com/questions/6947413/how-to-open-read-and-write-from-serial-port-in-c
		// and source code of busybox/microcom.c
		initSerialPort( ArduinoFd );

		// set up FD monitor
		// note: this doesn't return on error, so no need to check for errors....
		ArduinoFdMonitor = le_fdMonitor_Create("Arduino"
											   ,ArduinoFd
											   ,arduinoFdMonitorHandler
											   ,POLLIN
											  );

		isInitialised = true;	

		// create status timer
		PollTimerHandle = le_timer_Create("ArduinoPollTimer");
		LE_WARN_IF( LE_OK != (le_timer_SetHandler( PollTimerHandle, pollTimerHandler )), "PollTimer SetHandler failed");
		
		pollTimerHandler( PollTimerHandle );	// call timer handler - fire of initial character and restart timer

	}
	while (0);
	
	if ( !isInitialised )
	{
		LE_FATAL("Error initializing Arduino Master!");
	}

	/* Remember that COMPONENT_INIT() must exit or the rest of the application will never run */
}

