/*
 * Copyright (C) 2016-2017 Renfell Engineering Pty Ltd - All Rights Reserved
 *
 * Use of this work is subject to license and solely for use with hardware
 * supplied by Renfell Engineering Pty Ltd.
 *
 * Proprietary and confidential.
 * This file or any part thereof may not be used, distributed, copied, or
 * modified for use for any purpose other than that for which it was
 * originally provided without first receiving written permission of the
 * copyright holder.
 *
 * This source code is provided 'as is' without warranty of any kind, either
 * expressed or implied by consumer legislation or otherwise, including but
 * not limited to the implied warranties of merchantability and/or fitness
 * for a particular purpose.
 *
 * The end user acknowledges that this software is a "work in progress", and
 * as such should not be relied upon in a commercial environment or other
 * situations where economic loss or harm to persons may arise due to a
 * failure to perform to the end users specific requirements.
 *
 * By use of this software you agree to the use of the above terms.
 *
 */

#include "legato.h"
#include "interfaces.h"	/* include auto-generated api interfaces from Component.cdef */

/**
 * Complex example demonstrating GPIO usage using Renfell GPIOLab IoT Card
 *
 * This application does the following:
 *	1. Enable the IoT Card by manipulating the nRESET GPIO
 *	2. Flash LED on GPIO1 using a 500mS Timer (Output using fixed repeating timer) 
 *	3. Use TACT Switches on GPIO2 and GPIO3 to increment/decrement interval for GPIO4 LED (Input with Event) 
 *	4. Flash LED on GPIO4 using value set using (Output using one shot timer with variable interval)
 *  5. Store GPIO4 timer period using ConfigTree API
 *
 * Requirements:
 * 	Renfell 'GPIOLab IoT' card
 * 	Sierra Wireless MangOH Red DV2 or DV3 board with legato V16.10.1 installed
 *
 * Configuration:
 *
 *
 * 1. insert Renfell 'GPIO Lab' card into mangOH IoT slot #0
 *    (the only slot on the mangOH Red board).
 *
 * 2. power up the mangOH board.
 *
 * 3. build the gpioLab1 project using 
 *  	make clean target
 *
 * 4. install the gpioLab1 app onto the mangOH board using
 *		make install
 *	  Note that the app is set to MANUAL start.
 *
 * 5. Open a ssh console window to the target and run the command:
 *	    logread -f
 *    to display the system log
 *
 * 6. Open a second ssh console window to the target and run the following command to start the app:
 *		app start gpioLab1
 */


/*
 * All I/O based around using IoT Slot ('Slot 0') on the mangOH Red board
 *
 * mangOH Red IoT Slot 0 GPIO pins:
 *
 *	IoT_GPIO1: WP GPIO42
 *	IoT_GPIO2: WP GPIO13
 *	IoT_GPIO3: WP GPIO7
 *	IoT_GPIO4: WP GPIO8
 *
 *  IoT Reset: WP GPIO2
 *
 * NOTE:
 * 	IoT_GPIO1, IoT_GPIO2, IoT_GPIO3 and IoT_GPIO4 are 'created' as instances of le_gpio using requires: api: stanza in Component.cdef
 *  and are mapped to actual pins in the ADEF file
 *
 */

//-------------------------------------------------------------------------------------------------
/*
 * global variables
 */
//-------------------------------------------------------------------------------------------------

// GPIO1 Blink Timer Variable
le_timer_Ref_t Gpio1BlinkTimer = NULL;


// change event handles - references to callback event functions when pin changes state
IoT_GPIO2_ChangeEventHandlerRef_t Gpio2ChangeEventHandle = NULL;
IoT_GPIO3_ChangeEventHandlerRef_t Gpio3ChangeEventHandle = NULL;

// variables for GPIO4 blink timer
const uint32_t    Gpio4BlinkDifference  = 20;	// mS
const uint32_t    Gpio4BlinkIntervalMin = 40;
const uint32_t    Gpio4BlinkIntervalMax = 2000;
volatile uint32_t Gpio4BlinkInterval    = 500;	// mS
le_timer_Ref_t    Gpio4BlinkTimer       = NULL;

// variables to manage GPIO4 blink timer auto-repeat keys

const uint32_t    Gpio4AutoDelay    = 500; 	// mS - time to wait after button is pressed before starting auto inc/dec
const uint32_t    Gpio4AutoRepeat   = 100;	// mS - how long between 'auto button presses'
volatile uint32_t Gpio4AutoInterval = 0;	// current timer repeat intervals 
le_timer_Ref_t    Gpio4AutoIncTimer = NULL;	// timer to manage auto repeat keypress
le_timer_Ref_t    Gpio4AutoDecTimer = NULL;	// timer to manage auto repeat keypress

// forward definitions

static void gpio4AutoIncTimerHandler(le_timer_Ref_t pTimerRef );
static void gpio4AutoDecTimerHandler(le_timer_Ref_t pTimerRef );

//-------------------------------------------------------------------------------------------------
/*
 * Helper Function to stop and remove a timer
 *
 * *** NOTE: This doesn't free any context variable
 */
//-------------------------------------------------------------------------------------------------

void stopAndDeleteTimer( le_timer_Ref_t *pTimerRef )
{
	if (*pTimerRef != NULL )
	{
		le_timer_Stop( *pTimerRef );
		le_timer_Delete( *pTimerRef );	// make sure this timer is gone
		*pTimerRef = NULL;
	}
}

//-------------------------------------------------------------------------------------------------
/*
 * Timer Handler to toggle GPIO1
 */
//-------------------------------------------------------------------------------------------------
void gpio1BlinkTimerHandler(le_timer_Ref_t pTimerRef )
{
	le_result_t   result = LE_OK;
	static uint8_t state = 0;			// static variable to hold state of GPIO


	if (state) { result = IoT_GPIO1_Activate();   state = 0; }
	else       { result = IoT_GPIO1_Deactivate(); state = 1; }

	if ( result != LE_OK )
		LE_INFO( "Timer: GPIO returns [%d]", result );

//	LE_INFO( "state= [%s]", ((state)?"HI":"LO") );

    return;
}

//-------------------------------------------------------------------------------------------------
/*
 * GPIO2 pin change event handler
 */
//-------------------------------------------------------------------------------------------------

static void gpio2ChangeEventHandler( bool pState, void *pCtx )
{
	LE_INFO( "GPIO2 Change Event: new state: (%d)[%s]", pState, ((pState==true)?"true":"false"));

	// ensure that both Auto Inc and Auto Dec timers are stopped and deleted
	// no matter what the key press state
	stopAndDeleteTimer( &Gpio4AutoIncTimer );
	stopAndDeleteTimer( &Gpio4AutoDecTimer );

	if ( pState == true )	// only increment when state is now true (set)
	{
		Gpio4BlinkInterval  += Gpio4BlinkDifference;		// increment blink interval - make SLOWER

		if ( Gpio4BlinkInterval > Gpio4BlinkIntervalMax )
		{
			Gpio4BlinkInterval = Gpio4BlinkIntervalMax;
		}
		LE_INFO( "GPIO2: update Gpio4BlinkInterval[%4u]", Gpio4BlinkInterval ); 

		// button down, start the Auto Increment process
		
		Gpio4AutoIncTimer = le_timer_Create("Gpio4AutoIncTimer");

		le_timer_SetHandler ( Gpio4AutoIncTimer, gpio4AutoIncTimerHandler );
		le_timer_SetMsInterval( Gpio4AutoIncTimer, Gpio4AutoDelay );	// first time in, wait a longer time
		le_timer_SetRepeat( Gpio4AutoIncTimer, 1 );	// run once - this is default but it doesn't hurt
		le_timer_Start( Gpio4AutoIncTimer );
	}
	return;
}

//-------------------------------------------------------------------------------------------------
/*
 * GPIO3 pin change event handler
 */
//-------------------------------------------------------------------------------------------------

static void gpio3ChangeEventHandler( bool pState, void *pCtx )
{
	LE_INFO( "GPIO3 Change Event: new state: (%d)[%s]", pState, ((pState==true)?"true":"false"));

	// ensure that both Auto Inc and Auto Dec timers are stopped and deleted
	// no matter what the key press state
	stopAndDeleteTimer( &Gpio4AutoIncTimer );
	stopAndDeleteTimer( &Gpio4AutoDecTimer );

	if ( pState == true )	// only decrement when state is now true (set)
	{
		Gpio4BlinkInterval  -= Gpio4BlinkDifference;		// decrement blink interval - make FASTER

		if (   ( Gpio4BlinkInterval < Gpio4BlinkIntervalMin )	// to small?
			|| ( Gpio4BlinkInterval > Gpio4BlinkIntervalMax ) )	// Gpio4BlinkInterval is unsigned: this catches strange wraparound state
		{
			Gpio4BlinkInterval = Gpio4BlinkIntervalMin;
		}
		LE_INFO( "GPIO3: update Gpio4BlinkInterval[%4u]", Gpio4BlinkInterval );
		// button down, start the Auto Increment process
		
		Gpio4AutoDecTimer = le_timer_Create("Gpio4AutoDecTimer");

		le_timer_SetHandler ( Gpio4AutoDecTimer, gpio4AutoDecTimerHandler );
		le_timer_SetMsInterval( Gpio4AutoDecTimer, Gpio4AutoDelay );	// first time in, wait a longer time
		le_timer_SetRepeat( Gpio4AutoDecTimer, 1 );	// run once - this is default but it doesn't hurt
		le_timer_Start( Gpio4AutoDecTimer );
		
	}
	return;
}

//-------------------------------------------------------------------------------------------------
/*
 * Timer Handler to manage auto Increment button press for GPIO4
 */
//-------------------------------------------------------------------------------------------------
static void gpio4AutoIncTimerHandler(le_timer_Ref_t pTimerRef )
{

	Gpio4BlinkInterval  += Gpio4BlinkDifference;		// increment blink interval - make SLOWER

	if ( Gpio4BlinkInterval > Gpio4BlinkIntervalMax )
	{
		Gpio4BlinkInterval = Gpio4BlinkIntervalMax;
	}
	LE_INFO( "GPIO2: Auto update Gpio4BlinkInterval[%4u]", Gpio4BlinkInterval ); 

	le_timer_SetMsInterval( pTimerRef, Gpio4AutoRepeat );	// set the new timer interval
	le_timer_Restart( pTimerRef );							// restart the timer

    return;
}

//-------------------------------------------------------------------------------------------------
/*
 * Timer Handler to manage auto Increment button press for GPIO4
 */
//-------------------------------------------------------------------------------------------------
static void gpio4AutoDecTimerHandler(le_timer_Ref_t pTimerRef )
{
	Gpio4BlinkInterval  -= Gpio4BlinkDifference;			// decrement blink interval - make FASTER

	if (   ( Gpio4BlinkInterval < Gpio4BlinkIntervalMin )	// too small?
		|| ( Gpio4BlinkInterval > Gpio4BlinkIntervalMax ) )	// Gpio4BlinkInterval is unsigned: this catches strange wraparound state
	{
		Gpio4BlinkInterval = Gpio4BlinkIntervalMin;
	}

	LE_INFO( "GPIO3: Auto update Gpio4BlinkInterval[%4u]", Gpio4BlinkInterval ); 

	le_timer_SetMsInterval( pTimerRef, Gpio4AutoRepeat );	// set the new timer interval
	le_timer_Restart( pTimerRef );							// restart the timer

    return;
}

//-------------------------------------------------------------------------------------------------
/*
 * Timer Handler to toggle GPIO4
 */
//-------------------------------------------------------------------------------------------------
static void gpio4BlinkTimerHandler(le_timer_Ref_t pTimerRef )
{
	le_result_t   result = LE_OK;
	static uint8_t state = 0;			// static variable to hold state of GPIO


	if (state) { result = IoT_GPIO4_Activate();   state = 0; }
	else       { result = IoT_GPIO4_Deactivate(); state = 1; }

	if ( result != LE_OK )
		LE_INFO( "Timer: GPIO returns [%d]", result );

//	LE_INFO( "state= [%s]", ((state)?"HI":"LO") );

	le_timer_SetMsInterval( pTimerRef, Gpio4BlinkInterval );	// set the new timer interval
	le_timer_Restart( pTimerRef );					// restart the timer

    return;
}

//-------------------------------------------------------------------------------------------------
/*
 * Catch the shutdown event so we can save the values to configTree
 */
//-------------------------------------------------------------------------------------------------

static void sigHandlerSigTerm( int pSigNum )
{
	LE_INFO("SIGTERM caught, closing Application");

	stopAndDeleteTimer( &Gpio1BlinkTimer );
	stopAndDeleteTimer( &Gpio4AutoIncTimer );
	stopAndDeleteTimer( &Gpio4AutoDecTimer );
	stopAndDeleteTimer( &Gpio4BlinkTimer );

	// save the current Gpio4Blink Interval to the config tree for next time 
    le_cfg_QuickSetInt("/Gpio4BlinkInterval", Gpio4BlinkInterval);
}

//-------------------------------------------------------------------------------------------------
/*
 * Component Entry and startup
 *
 * NOTE: this function MUST complete for otherwise the rest of the app will not run
 */
//-------------------------------------------------------------------------------------------------

COMPONENT_INIT
{
	le_result_t result = LE_OK;

    LE_INFO( "complexBlinky Init" );

    // setup to catch application termination and shutdown cleanly
    le_sig_Block( SIGTERM );
    le_sig_SetEventHandler( SIGTERM, sigHandlerSigTerm );

    // ensure that IoT slot is 'Enabled' 
	LE_FATAL_IF( (LE_OK != (result = IoT_RESET_SetPushPullOutput(  IOT_RESET_ACTIVE_HIGH, true ) )),
							"GPIO IOT_RESET SetPushPull() failed: [%d]%s", result, LE_RESULT_TXT(result));

	// configure GPIO1 as output - use IoT slot 0, GPIO1 (WP85 pin GPIO42)
    result = IoT_GPIO1_SetPushPullOutput( IOT_GPIO1_ACTIVE_HIGH, true ); // constant renamed from LE_GPIO_ACTIVE_HIGH via binding
    if ( result != LE_OK )
    {
    	LE_WARN("GPIO1: SetPushPullOutput returned (%d)[%s]", result, LE_RESULT_TXT(result) );
    }

	// configure GPIO2 as input - IoT slot 0, GPIO2 (wp85 pin GPIO13)
	// input with active low polarity and internal pullup resistor
	LE_WARN_IF( (LE_OK != (result = IoT_GPIO2_SetInput( IOT_GPIO2_ACTIVE_LOW ))),
								"GPIO2: SetInput returned ((%d)[%s]", result, LE_RESULT_TXT(result) ) ;
	LE_WARN_IF( (LE_OK != (result = IoT_GPIO2_EnablePullUp( ))),
								"GPIO2: EnablePullUp returned ((%d)[%s]", result, LE_RESULT_TXT(result) ) ;


	// configure GPIO3 as input - IoT slot 0, GPIO3 (wp85 pin GPIO7)
	// input with active high polarity and internal pullup resistor
	LE_WARN_IF( (LE_OK != (result = IoT_GPIO3_SetInput( IOT_GPIO3_ACTIVE_HIGH ))),
								"GPIO3: SetInput returned ((%d)[%s]", result, LE_RESULT_TXT(result) ) ;
	LE_WARN_IF( (LE_OK != (result = IoT_GPIO3_EnablePullUp( ))),
								"GPIO3: EnablePullUp returned ((%d)[%s]", result, LE_RESULT_TXT(result) ) ;


	// add change handler to GPIO3 and GPIO4
	// change on both edge
	Gpio2ChangeEventHandle = IoT_GPIO2_AddChangeEventHandler( IOT_GPIO2_EDGE_BOTH, gpio2ChangeEventHandler, NULL, 20 );
	Gpio3ChangeEventHandle = IoT_GPIO3_AddChangeEventHandler( IOT_GPIO3_EDGE_BOTH, gpio3ChangeEventHandler, NULL, 20 );

	// configure GPIO4 as output - IoT slot 0, GPIO4 (WP85 pin GPIO8)
	LE_WARN_IF( (LE_OK != (result = IoT_GPIO4_SetPushPullOutput( IOT_GPIO4_ACTIVE_HIGH, true))), 
								"GPIO4: SetPushPullOutput returned (%d)[%s]", result, LE_RESULT_TXT(result) ) ;


	// set up timer to blink GPIO1 (roughly) every 500 milliseconds
    Gpio1BlinkTimer = le_timer_Create("Gpio1BlinkTimer");

    le_timer_SetHandler ( Gpio1BlinkTimer, gpio1BlinkTimerHandler );
    le_timer_SetMsInterval( Gpio1BlinkTimer, 500 );
	le_timer_SetRepeat( Gpio1BlinkTimer, 0);	// run forever
    le_timer_Start( Gpio1BlinkTimer );


	// set up timer to blink GPIO4 depending on value set using buttons

	// read initial value from configTree
	// if not set, use the default value
    Gpio4BlinkInterval = le_cfg_QuickGetInt("/Gpio4BlinkInterval", Gpio4BlinkInterval);

	LE_INFO( "Gpio4BlinkInterval: [%u]", Gpio4BlinkInterval );
    Gpio4BlinkTimer = le_timer_Create("Gpio4BlinkTimer");

    le_timer_SetHandler ( Gpio4BlinkTimer, gpio4BlinkTimerHandler );
    le_timer_SetMsInterval( Gpio4BlinkTimer, Gpio4BlinkInterval );
	le_timer_SetRepeat( Gpio4BlinkTimer, 1 );	// set the timer to run once - this is default but it doesn't hurt
	gpio4BlinkTimerHandler( Gpio4BlinkTimer );	// call the handler to set up the LED & restart the timer as required

}

